// -*- tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
// vi: set et ts=4 sw=2 sts=2:
#ifndef DUNE_FUNCTIONS_COMMON_POLYMORPHICSMALLOBJECT_HH
#define DUNE_FUNCTIONS_COMMON_POLYMORPHICSMALLOBJECT_HH

#include <utility>
#include <type_traits>

#include <dune/common/std/type_traits.hh>
#include <dune/common/hybridutilities.hh>

namespace Dune {
namespace Functions {


/**
 * \brief A wrapper providing small object optimization with polymorphic types
 *
 * \ingroup Utility
 * \ingroup TypeErasure
 *
 * \tparam Base Base class type of wrapped objects
 * \tparam bufferSize Size of small object buffer
 *
 * This class encapsulates small object optimization for polymorphic types.
 * The type of objects passed to the constructor must be derived from the
 * base class type Base.
 *
 * If the size of the derived type fits into the static buffer, then the
 * wrapped object is stored there, otherwise it is allocated dynamically.
 *
 * Notice that this class does implement use type erasure for destructors,
 * copy/move constructors and copy/move assignment. Hence it requires
 * that Base has a virtual destructor.
 *
 * In order to make the copy constructor work for polymorphic types,
 * Base must provide virtual methods `Base* clone()` and `Base* clone(void*)`.
 * The former should return a pointer to a dynamically allocated clone, while
 * the latter should call the appropriate placement-new with the passed pointer.
 *
 * Similarly the polymorphic type has to implement a virtual `Base* move(void*)`
 * method.
 * This should call placement-new and can std::move all the
 * data but leave the object in a valid and probably unusable state.
 */
template<class Base, size_t bufferSize>
class PolymorphicSmallObject
{
public:

  //! Default constructor
  PolymorphicSmallObject() :
    p_(nullptr)
  {}

  /**
   * \brief Construct from object
   *
   * \tparam Derived Type of object to be stored, must be derived from Base
   * \param derived Object to be stored
   */
  template<class Derived,
        typename std::enable_if<std::is_base_of<Base, std::remove_cv_t<
          std::remove_reference_t<Derived>>>::value, int>::type = 0>
  PolymorphicSmallObject(Derived&& derived)
  {
    using namespace Dune::Hybrid;
    auto useBuffer = Dune::Std::bool_constant<(sizeof(Derived) <= bufferSize)>();
    ifElse(useBuffer, [&](auto id) {
      p_ = new (&buffer_) Derived(std::forward<Derived>(derived));
    }, [&](auto id) {
      p_ = new Derived(std::forward<Derived>(derived));
    });
  }

  //! Move constructor from other PolymorphicSmallObject
  PolymorphicSmallObject(PolymorphicSmallObject&& other)
  {
    moveToWrappedObject(std::move(other));
  }

  //! Copy constructor from other PolymorphicSmallObject
  PolymorphicSmallObject(const PolymorphicSmallObject& other)
  {
    copyToWrappedObject(other);
  }

  //! Destructor
  ~PolymorphicSmallObject()
  {
    destroyWrappedObject();
  }

  //! Copy assignment from other PolymorphicSmallObject
  PolymorphicSmallObject& operator=(const PolymorphicSmallObject& other)
  {
    if (&other!=this)
    {
      destroyWrappedObject();
      copyToWrappedObject(other);
    }
    return *this;
  }

  //! Move assignment from other PolymorphicSmallObject
  PolymorphicSmallObject& operator=(PolymorphicSmallObject&& other)
  {
    destroyWrappedObject();
    moveToWrappedObject(std::move(other));
    return *this;
  }

  //! Check if *this is not empty
  explicit operator bool() const
  {
    return p_;
  }

  //! Check if object is stored in internal stack buffer
  bool bufferUsed() const
  {
    return ((void*) (p_) == (void*)(&buffer_));
  }

  //! Obtain reference to stored object
  const Base& get() const
  {
    return *p_;
  }

  //! Obtain mutable reference to stored object
  Base& get()
  {
    return *p_;
  }

private:

  void destroyWrappedObject()
  {
    if (operator bool())
    {
      if (bufferUsed())
        p_->~Base();
      else
        delete p_;
    }
  }

  void moveToWrappedObject(PolymorphicSmallObject&& other)
  {
    if (other.bufferUsed())
      p_ = other.p_->move(&buffer_);
    else
    {
      // We don't need to check for &other_!=this, because you can't
      // have an rvalue to *this and call it's assignment/constructor
      // at the same time. (Despite trying to shoot yourself in the foot
      // with std::move explicitly.)

      // Take ownership of allocated object
      p_ = other.p_;

      // Leave pointer in a clean state to avoid double freeing it.
      other.p_ = 0;
    }
  }

  void copyToWrappedObject(const PolymorphicSmallObject& other)
  {
    if (other.bufferUsed())
      p_ = other.p_->clone(&buffer_);
    else
      p_ = other.p_->clone();
  }

  std::aligned_storage_t<bufferSize> buffer_;
  Base* p_;
};


} // namespace Functions
} // namespace Dune

#endif // DUNE_FUNCTIONS_COMMON_POLYMORPHICSMALLOBJECT_HH
