//
// ========================================================================
// Copyright (c) 1995 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0, or the Apache License, Version 2.0
// which is available at https://www.apache.org/licenses/LICENSE-2.0.
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.ee9.websocket.client.impl;

import java.net.HttpCookie;
import java.net.URI;
import java.security.Principal;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.eclipse.jetty.ee9.websocket.api.ExtensionConfig;
import org.eclipse.jetty.ee9.websocket.api.UpgradeRequest;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpFields;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpScheme;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.util.MultiMap;
import org.eclipse.jetty.util.UrlEncoded;
import org.eclipse.jetty.websocket.core.client.CoreClientUpgradeRequest;

import static java.nio.charset.StandardCharsets.UTF_8;

/**
 * Representing the Jetty {@link org.eclipse.jetty.client.Request}
 * in the {@link UpgradeRequest} interface.
 */
public class DelegatedJettyClientUpgradeRequest implements UpgradeRequest
{
    private final CoreClientUpgradeRequest delegate;
    private final Map<String, List<String>> headers;

    public DelegatedJettyClientUpgradeRequest(CoreClientUpgradeRequest delegate)
    {
        this.delegate = delegate;
        this.headers = HttpFields.asMap(delegate.getHeaders());
    }

    @Override
    public List<HttpCookie> getCookies()
    {
        return delegate.getCookies().stream()
            .map(org.eclipse.jetty.http.HttpCookie::asJavaNetHttpCookie)
            .toList();
    }

    @Override
    public String getHeader(String name)
    {
        return delegate.getHeaders().get(name);
    }

    @Override
    public int getHeaderInt(String name)
    {
        HttpField field = delegate.getHeaders().getField(name);
        if (field == null)
            return -1;
        return field.getIntValue();
    }

    @Override
    public List<String> getHeaders(String name)
    {
        return delegate.getHeaders().getValuesList(name);
    }

    @Override
    public Map<String, List<String>> getHeaders()
    {
        return headers;
    }

    @Override
    public String getHost()
    {
        return delegate.getURI().getHost();
    }

    @Override
    public String getHttpVersion()
    {
        return delegate.getVersion().asString();
    }

    public void configure(EndPoint endpoint)
    {
    }

    @Override
    public String getMethod()
    {
        return delegate.getMethod();
    }

    @Override
    public String getOrigin()
    {
        return delegate.getHeaders().get(HttpHeader.ORIGIN);
    }

    @Override
    public Map<String, List<String>> getParameterMap()
    {
        if (getQueryString() == null)
            return Collections.emptyMap();

        MultiMap<String> params = new MultiMap<>();
        UrlEncoded.decodeTo(getQueryString(), params, UTF_8);
        return params;
    }

    @Override
    public String getProtocolVersion()
    {
        return delegate.getHeaders().get(HttpHeader.SEC_WEBSOCKET_VERSION);
    }

    @Override
    public String getQueryString()
    {
        return delegate.getURI().getRawQuery();
    }

    @Override
    public URI getRequestURI()
    {
        return delegate.getURI();
    }

    @Override
    public List<String> getSubProtocols()
    {
        return delegate.getHeaders().getValuesList(HttpHeader.SEC_WEBSOCKET_SUBPROTOCOL);
    }

    @Override
    public boolean hasSubProtocol(String test)
    {
        HttpField field = delegate.getHeaders().getField(HttpHeader.SEC_WEBSOCKET_SUBPROTOCOL);
        if (field == null)
            return false;
        return field.contains(test);
    }

    @Override
    public boolean isSecure()
    {
        return HttpScheme.isSecure(delegate.getURI().getScheme());
    }

    @Override
    public Principal getUserPrincipal()
    {
        return null;
    }

    @Override
    public List<ExtensionConfig> getExtensions()
    {
        List<String> rawExtensions = delegate.getHeaders().getValuesList(HttpHeader.SEC_WEBSOCKET_EXTENSIONS);
        if (rawExtensions == null || rawExtensions.isEmpty())
            return Collections.emptyList();

        return rawExtensions.stream().map(ExtensionConfig::parse).collect(Collectors.toList());
    }
}
