/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/kumaraswamy/cdf' );
import Kumaraswamy = require( './../../../../../base/dists/kumaraswamy/ctor' );
import kurtosis = require( './../../../../../base/dists/kumaraswamy/kurtosis' );
import logcdf = require( './../../../../../base/dists/kumaraswamy/logcdf' );
import logpdf = require( './../../../../../base/dists/kumaraswamy/logpdf' );
import mean = require( './../../../../../base/dists/kumaraswamy/mean' );
import median = require( './../../../../../base/dists/kumaraswamy/median' );
import mode = require( './../../../../../base/dists/kumaraswamy/mode' );
import pdf = require( './../../../../../base/dists/kumaraswamy/pdf' );
import quantile = require( './../../../../../base/dists/kumaraswamy/quantile' );
import skewness = require( './../../../../../base/dists/kumaraswamy/skewness' );
import stdev = require( './../../../../../base/dists/kumaraswamy/stdev' );
import variance = require( './../../../../../base/dists/kumaraswamy/variance' );

/**
* Interface describing the `kumaraswamy` namespace.
*/
interface Namespace {
	/**
	* Kumaraswamy's double bounded distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 0.5, 1.0, 1.0 );
	* // returns 0.5
	*
	* y = ns.cdf( 0.5, 2.0, 4.0 );
	* // returns ~0.684
	*
	* var mycdf = ns.cdf.factory( 0.5, 0.5 );
	*
	* y = mycdf( 0.8 );
	* // returns ~0.675
	*
	* y = mycdf( 0.3 );
	* // returns ~0.327
	*/
	cdf: typeof cdf;

	/**
	* Kumaraswamy double bounded distribution.
	*/
	Kumaraswamy: typeof Kumaraswamy;

	/**
	* Returns the excess kurtosis of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns excess kurtosis
	*
	* @example
	* var v = ns.kurtosis( 0.5, 1.0 );
	* // returns ~2.143
	*
	* @example
	* var v = ns.kurtosis( 4.0, 12.0 );
	* // returns ~2.704
	*
	* @example
	* var v = ns.kurtosis( 12.0, 2.0 );
	* // returns ~4.817
	*
	* @example
	* var v = ns.kurtosis( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.kurtosis( NaN, 2.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Kumaraswamy's double bounded distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 0.5, 1.0, 1.0 );
	* // returns ~-0.693
	*
	* y = ns.logcdf( 0.5, 2.0, 4.0 );
	* // returns ~-0.38
	*
	* var mylogcdf = ns.logcdf.factory( 0.5, 0.5 );
	*
	* y = mylogcdf( 0.8 );
	* // returns ~-0.393
	*
	* y = mylogcdf( 0.3 );
	* // returns ~-1.118
	*/
	logcdf: typeof logcdf;

	/**
	* Arcsine distribution logarithm of probability density function (PDF).
	*
	* @param x - input value
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 0.5, 1.0, 1.0 );
	* // returns 0.0
	*
	* y = ns.logpdf( 0.5, 2.0, 4.0 );
	* // returns ~0.523
	*
	* var mylogpdf = ns.logpdf.factory( 0.5, 0.5 );
	*
	* y = mylogpdf( 0.8 );
	* // returns ~-0.151
	*
	* y = mylogpdf( 0.3 );
	* // returns ~-0.387
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns expected value
	*
	* @example
	* var v = ns.mean( 1.5, 1.5 );
	* // returns ~0.512
	*
	* @example
	* var v = ns.mean( 4.0, 12.0 );
	* // returns ~0.481
	*
	* @example
	* var v = ns.mean( 12.0, 2.0 );
	* // returns ~0.886
	*
	* @example
	* var v = ns.mean( 1.5, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( -0.1, 1.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mean( NaN, 2.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns median
	*
	* @example
	* var v = ns.median( 0.5, 1.0 );
	* // returns 0.25
	*
	* @example
	* var v = ns.median( 4.0, 12.0 );
	* // returns ~0.487
	*
	* @example
	* var v = ns.median( 12.0, 2.0 );
	* // returns ~0.903
	*
	* @example
	* var v = ns.median( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.median( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.median( NaN, 2.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Returns the mode of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns mode
	*
	* @example
	* var v = ns.mode( 1.5, 1.5 );
	* // returns ~0.543
	*
	* @example
	* var v = ns.mode( 4.0, 12.0 );
	* // returns ~0.503
	*
	* @example
	* var v = ns.mode( 12.0, 2.0 );
	* // returns ~0.94
	*
	* @example
	* var v = ns.mode( 1.0, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 1.5, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( -0.1, 1.5 );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.mode( NaN, 2.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Kumaraswamy's double bounded distribution probability density function (PDF).
	*
	* @param x - input value
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 0.5, 1.0, 1.0 );
	* // returns 1.0
	*
	* y = ns.pdf( 0.5, 2.0, 4.0 );
	* // returns ~1.688
	*
	* var mypdf = ns.pdf.factory( 0.5, 0.5 );
	*
	* y = mypdf( 0.8 );
	* // returns ~0.86
	*
	* y = mypdf( 0.3 );
	* // returns ~0.679
	*/
	pdf: typeof pdf;

	/**
	* Kumaraswamy's double bounded distribution quantile function.
	*
	* @param x - input value
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.5, 1.0, 1.0 );
	* // returns 0.5
	*
	* y = ns.quantile( 0.5, 2.0, 4.0 );
	* // returns ~0.399
	*
	* var myQuantile = factory( 0.5, 0.5 );
	*
	* y = myQuantile( 0.8 );
	* // returns ~0.922
	*
	* y = myQuantile( 0.3 );
	* // returns ~0.26
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns skewness
	*
	* @example
	* var v = ns.skewness( 0.5, 1.0 );
	* // returns ~0.639
	*
	* @example
	* var v = ns.skewness( 4.0, 12.0 );
	* // returns ~-0.201
	*
	* @example
	* var v = ns.skewness( 12.0, 2.0 );
	* // returns ~-1.2
	*
	* @example
	* var v = ns.skewness( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.skewness( NaN, 2.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns standard deviation
	*
	* @example
	* var v = ns.stdev( 0.5, 1.0 );
	* // returns ~0.298
	*
	* @example
	* var v = ns.stdev( 4.0, 12.0 );
	* // returns ~0.13
	*
	* @example
	* var v = ns.stdev( 12.0, 2.0 );
	* // returns ~0.077
	*
	* @example
	* var v = ns.stdev( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.stdev( NaN, 2.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance of a Kumaraswamy's double bounded distribution.
	*
	* ## Notes
	*
	* -   If `a <= 0` or `b <= 0`, the function returns `NaN`.
	*
	* @param a - first shape parameter
	* @param b - second shape parameter
	* @returns variance
	*
	* @example
	* var v = ns.variance( 0.5, 1.0 );
	* // returns ~0.089
	*
	* @example
	* var v = ns.variance( 4.0, 12.0 );
	* // returns ~0.017
	*
	* @example
	* var v = ns.variance( 12.0, 2.0 );
	* // returns ~0.006
	*
	* @example
	* var v = ns.variance( 1.0, -0.1 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( -0.1, 1.0 );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( 2.0, NaN );
	* // returns NaN
	*
	* @example
	* var v = ns.variance( NaN, 2.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Kumaraswamy's double bounded distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
